/*
 * Decompiled with CFR 0.152.
 */
package be.tarsos.dsp;

import be.tarsos.dsp.AudioEvent;
import be.tarsos.dsp.AudioProcessor;
import be.tarsos.dsp.io.TarsosDSPAudioFloatConverter;
import be.tarsos.dsp.io.TarsosDSPAudioFormat;
import be.tarsos.dsp.io.TarsosDSPAudioInputStream;
import java.io.IOException;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.logging.Level;
import java.util.logging.Logger;

public class AudioDispatcher
implements Runnable {
    private static final Logger LOG = Logger.getLogger(AudioDispatcher.class.getName());
    private final TarsosDSPAudioInputStream audioInputStream;
    private float[] audioFloatBuffer;
    private byte[] audioByteBuffer;
    private final List<AudioProcessor> audioProcessors = new CopyOnWriteArrayList<AudioProcessor>();
    private final TarsosDSPAudioFloatConverter converter;
    private final TarsosDSPAudioFormat format;
    private int floatOverlap;
    private int floatStepSize;
    private int byteOverlap;
    private int byteStepSize;
    private long bytesToSkip;
    private long bytesProcessed;
    private AudioEvent audioEvent;
    private boolean stopped;
    private boolean zeroPadFirstBuffer;
    private boolean zeroPadLastBuffer;

    public AudioDispatcher(TarsosDSPAudioInputStream stream, int audioBufferSize, int bufferOverlap) {
        this.audioInputStream = stream;
        this.format = this.audioInputStream.getFormat();
        this.setStepSizeAndOverlap(audioBufferSize, bufferOverlap);
        this.audioEvent = new AudioEvent(this.format);
        this.audioEvent.setFloatBuffer(this.audioFloatBuffer);
        this.audioEvent.setOverlap(bufferOverlap);
        this.converter = TarsosDSPAudioFloatConverter.getConverter(this.format);
        this.stopped = false;
        this.bytesToSkip = 0L;
        this.zeroPadLastBuffer = true;
    }

    public void skip(double seconds) {
        this.bytesToSkip = Math.round(seconds * (double)this.format.getSampleRate()) * (long)this.format.getFrameSize();
    }

    public void setStepSizeAndOverlap(int audioBufferSize, int bufferOverlap) {
        this.audioFloatBuffer = new float[audioBufferSize];
        this.floatOverlap = bufferOverlap;
        this.floatStepSize = this.audioFloatBuffer.length - this.floatOverlap;
        this.audioByteBuffer = new byte[this.audioFloatBuffer.length * this.format.getFrameSize()];
        this.byteOverlap = this.floatOverlap * this.format.getFrameSize();
        this.byteStepSize = this.floatStepSize * this.format.getFrameSize();
    }

    public void setZeroPadFirstBuffer(boolean zeroPadFirstBuffer) {
        this.zeroPadFirstBuffer = zeroPadFirstBuffer;
    }

    public void setZeroPadLastBuffer(boolean zeroPadLastBuffer) {
        this.zeroPadLastBuffer = zeroPadLastBuffer;
    }

    public void addAudioProcessor(AudioProcessor audioProcessor) {
        this.audioProcessors.add(audioProcessor);
        LOG.fine("Added an audioprocessor to the list of processors: " + audioProcessor.toString());
    }

    public void removeAudioProcessor(AudioProcessor audioProcessor) {
        this.audioProcessors.remove(audioProcessor);
        audioProcessor.processingFinished();
        LOG.fine("Remove an audioprocessor to the list of processors: " + audioProcessor.toString());
    }

    @Override
    public void run() {
        int bytesRead = 0;
        if (this.bytesToSkip != 0L) {
            this.skipToStart();
        }
        try {
            this.audioEvent.setBytesProcessed(this.bytesProcessed);
            bytesRead = this.readNextAudioBlock();
        }
        catch (IOException e) {
            String message = "Error while reading audio input stream: " + e.getMessage();
            LOG.warning(message);
            throw new Error(message);
        }
        while (bytesRead != 0 && !this.stopped) {
            for (AudioProcessor processor : this.audioProcessors) {
                if (!processor.process(this.audioEvent)) break;
            }
            if (this.stopped) continue;
            this.bytesProcessed += (long)bytesRead;
            this.audioEvent.setBytesProcessed(this.bytesProcessed);
            try {
                bytesRead = this.readNextAudioBlock();
                this.audioEvent.setOverlap(this.floatOverlap);
            }
            catch (IOException e) {
                String message = "Error while reading audio input stream: " + e.getMessage();
                LOG.warning(message);
                throw new Error(message);
            }
        }
        if (!this.stopped) {
            this.stop();
        }
    }

    private void skipToStart() {
        long skipped = 0L;
        try {
            skipped = this.audioInputStream.skip(this.bytesToSkip);
            if (skipped != this.bytesToSkip) {
                throw new IOException();
            }
            this.bytesProcessed += this.bytesToSkip;
        }
        catch (IOException e) {
            String message = String.format("Did not skip the expected amount of bytes,  %d skipped, %d expected!", skipped, this.bytesToSkip);
            LOG.warning(message);
            throw new Error(message);
        }
    }

    public void stop() {
        this.stopped = true;
        for (AudioProcessor processor : this.audioProcessors) {
            processor.processingFinished();
        }
        try {
            this.audioInputStream.close();
        }
        catch (IOException e) {
            LOG.log(Level.SEVERE, "Closing audio stream error.", e);
        }
    }

    private int readNextAudioBlock() throws IOException {
        int offsetInSamples;
        int offsetInBytes;
        int bytesToRead;
        boolean isFirstBuffer;
        assert (this.floatOverlap < this.audioFloatBuffer.length);
        boolean bl = isFirstBuffer = this.bytesProcessed == 0L || this.bytesProcessed == this.bytesToSkip;
        if (isFirstBuffer && !this.zeroPadFirstBuffer) {
            bytesToRead = this.audioByteBuffer.length;
            offsetInBytes = 0;
            offsetInSamples = 0;
        } else {
            bytesToRead = this.byteStepSize;
            offsetInBytes = this.byteOverlap;
            offsetInSamples = this.floatOverlap;
        }
        if (!isFirstBuffer && this.audioFloatBuffer.length == this.floatOverlap + this.floatStepSize) {
            System.arraycopy(this.audioFloatBuffer, this.floatStepSize, this.audioFloatBuffer, 0, this.floatOverlap);
        }
        int totalBytesRead = 0;
        int bytesRead = 0;
        boolean endOfStream = false;
        while (!this.stopped && !endOfStream && totalBytesRead < bytesToRead) {
            try {
                bytesRead = this.audioInputStream.read(this.audioByteBuffer, offsetInBytes + totalBytesRead, bytesToRead - totalBytesRead);
            }
            catch (IndexOutOfBoundsException e) {
                bytesRead = -1;
            }
            if (bytesRead == -1) {
                endOfStream = true;
                continue;
            }
            totalBytesRead += bytesRead;
        }
        if (endOfStream) {
            if (this.zeroPadLastBuffer) {
                for (int i = offsetInBytes + totalBytesRead; i < this.audioByteBuffer.length; ++i) {
                    this.audioByteBuffer[i] = 0;
                }
                this.converter.toFloatArray(this.audioByteBuffer, offsetInBytes, this.audioFloatBuffer, offsetInSamples, this.floatStepSize);
            } else {
                byte[] audioByteBufferContent = this.audioByteBuffer;
                this.audioByteBuffer = new byte[offsetInBytes + totalBytesRead];
                for (int i = 0; i < this.audioByteBuffer.length; ++i) {
                    this.audioByteBuffer[i] = audioByteBufferContent[i];
                }
                int totalSamplesRead = totalBytesRead / this.format.getFrameSize();
                this.audioFloatBuffer = new float[offsetInSamples + totalBytesRead / this.format.getFrameSize()];
                this.converter.toFloatArray(this.audioByteBuffer, offsetInBytes, this.audioFloatBuffer, offsetInSamples, totalSamplesRead);
            }
        } else if (bytesToRead == totalBytesRead) {
            if (isFirstBuffer && !this.zeroPadFirstBuffer) {
                this.converter.toFloatArray(this.audioByteBuffer, 0, this.audioFloatBuffer, 0, this.audioFloatBuffer.length);
            } else {
                this.converter.toFloatArray(this.audioByteBuffer, offsetInBytes, this.audioFloatBuffer, offsetInSamples, this.floatStepSize);
            }
        } else if (!this.stopped) {
            throw new IOException(String.format("The end of the audio stream has not been reached and the number of bytes read (%d) is not equal to the expected amount of bytes(%d).", totalBytesRead, bytesToRead));
        }
        this.audioEvent.setFloatBuffer(this.audioFloatBuffer);
        this.audioEvent.setOverlap(offsetInSamples);
        return totalBytesRead;
    }

    public TarsosDSPAudioFormat getFormat() {
        return this.format;
    }

    public float secondsProcessed() {
        return (float)(this.bytesProcessed / (long)(this.format.getSampleSizeInBits() / 8)) / this.format.getSampleRate() / (float)this.format.getChannels();
    }

    public void setAudioFloatBuffer(float[] audioBuffer) {
        this.audioFloatBuffer = audioBuffer;
    }

    public boolean isStopped() {
        return this.stopped;
    }
}

