/*
 * Decompiled with CFR 0.152.
 */
package be.tarsos.dsp;

import be.tarsos.dsp.AudioEvent;
import be.tarsos.dsp.AudioProcessor;
import be.tarsos.dsp.util.PitchConverter;
import be.tarsos.dsp.util.fft.FFT;
import be.tarsos.dsp.util.fft.HammingWindow;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

public class SpectralPeakProcessor
implements AudioProcessor {
    private final int sampleRate;
    private final double dt;
    private final double cbin;
    private final double inv_2pi;
    private final double inv_deltat;
    private final double inv_2pideltat;
    private final FFT fft;
    private final float[] currentPhaseOffsets;
    private final float[] magnitudes;
    private final float[] frequencyEstimates;
    private float[] previousPhaseOffsets;

    public SpectralPeakProcessor(int bufferSize, int overlap, int sampleRate) {
        this.fft = new FFT(bufferSize, new HammingWindow());
        this.magnitudes = new float[bufferSize / 2];
        this.currentPhaseOffsets = new float[bufferSize / 2];
        this.frequencyEstimates = new float[bufferSize / 2];
        this.dt = (double)(bufferSize - overlap) / (double)sampleRate;
        this.cbin = this.dt * (double)sampleRate / (double)bufferSize;
        this.inv_2pi = 0.15915494309189535;
        this.inv_deltat = 1.0 / this.dt;
        this.inv_2pideltat = this.inv_deltat * this.inv_2pi;
        this.sampleRate = sampleRate;
    }

    private void calculateFFT(float[] audio) {
        float[] fftData = (float[])audio.clone();
        this.fft.powerPhaseFFT(fftData, this.magnitudes, this.currentPhaseOffsets);
    }

    private void normalizeMagintudes() {
        int i;
        float maxMagnitude = -1000000.0f;
        for (i = 0; i < this.magnitudes.length; ++i) {
            maxMagnitude = Math.max(maxMagnitude, this.magnitudes[i]);
        }
        for (i = 1; i < this.magnitudes.length; ++i) {
            this.magnitudes[i] = (float)(10.0 * Math.log10(this.magnitudes[i] / maxMagnitude)) + 75.0f;
        }
    }

    @Override
    public boolean process(AudioEvent audioEvent) {
        float[] audio = audioEvent.getFloatBuffer();
        this.calculateFFT(audio);
        this.calculateFrequencyEstimates();
        this.normalizeMagintudes();
        this.previousPhaseOffsets = (float[])this.currentPhaseOffsets.clone();
        return true;
    }

    @Override
    public void processingFinished() {
    }

    private void calculateFrequencyEstimates() {
        for (int i = 0; i < this.frequencyEstimates.length; ++i) {
            this.frequencyEstimates[i] = this.getFrequencyForBin(i);
        }
    }

    public float[] getMagnitudes() {
        return (float[])this.magnitudes.clone();
    }

    public float[] getFrequencyEstimates() {
        return (float[])this.frequencyEstimates.clone();
    }

    private float getFrequencyForBin(int binIndex) {
        float frequencyInHertz;
        if (this.previousPhaseOffsets != null) {
            float phaseDelta = this.currentPhaseOffsets[binIndex] - this.previousPhaseOffsets[binIndex];
            long k = Math.round(this.cbin * (double)binIndex - this.inv_2pi * (double)phaseDelta);
            frequencyInHertz = (float)(this.inv_2pideltat * (double)phaseDelta + this.inv_deltat * (double)k);
        } else {
            frequencyInHertz = (float)this.fft.binToHz(binIndex, this.sampleRate);
        }
        return frequencyInHertz;
    }

    public static float[] calculateNoiseFloor(float[] magnitudes, int medianFilterLength, float noiseFloorFactor) {
        float[] noisefloor = new float[magnitudes.length];
        float median = (float)SpectralPeakProcessor.median((float[])magnitudes.clone());
        for (int i = 0; i < magnitudes.length; ++i) {
            double[] noiseFloorBuffer = new double[medianFilterLength];
            int index = 0;
            for (int j = i - medianFilterLength / 2; j <= i + medianFilterLength / 2 && index < noiseFloorBuffer.length; ++index, ++j) {
                noiseFloorBuffer[index] = j >= 0 && j < magnitudes.length ? (double)magnitudes[j] : (double)median;
            }
            noisefloor[i] = SpectralPeakProcessor.median(noiseFloorBuffer) * noiseFloorFactor;
        }
        float rampLength = 12.0f;
        int i = 0;
        while ((float)i <= rampLength) {
            float ramp = 1.0f;
            ramp = (float)(-1.0 * Math.log((float)i / rampLength)) + 1.0f;
            noisefloor[i] = ramp * noisefloor[i];
            ++i;
        }
        return noisefloor;
    }

    public static List<Integer> findLocalMaxima(float[] magnitudes, float[] noisefloor) {
        ArrayList<Integer> localMaximaIndexes = new ArrayList<Integer>();
        for (int i = 1; i < magnitudes.length - 1; ++i) {
            boolean largerThanNoiseFloor;
            boolean largerThanPrevious = magnitudes[i - 1] < magnitudes[i];
            boolean largerThanNext = magnitudes[i] > magnitudes[i + 1];
            boolean bl = largerThanNoiseFloor = magnitudes[i] > noisefloor[i];
            if (!largerThanPrevious || !largerThanNext || !largerThanNoiseFloor) continue;
            localMaximaIndexes.add(i);
        }
        return localMaximaIndexes;
    }

    private static int findMaxMagnitudeIndex(float[] magnitudes) {
        int maxMagnitudeIndex = 0;
        float maxMagnitude = -1000000.0f;
        for (int i = 1; i < magnitudes.length - 1; ++i) {
            if (!(magnitudes[i] > maxMagnitude)) continue;
            maxMagnitude = magnitudes[i];
            maxMagnitudeIndex = i;
        }
        return maxMagnitudeIndex;
    }

    public static List<SpectralPeak> findPeaks(float[] magnitudes, float[] frequencyEstimates, List<Integer> localMaximaIndexes, int numberOfPeaks, int minDistanceInCents) {
        int i;
        int maxMagnitudeIndex = SpectralPeakProcessor.findMaxMagnitudeIndex(magnitudes);
        ArrayList<SpectralPeak> spectralPeakList = new ArrayList<SpectralPeak>();
        if (localMaximaIndexes.size() == 0) {
            return spectralPeakList;
        }
        float referenceFrequency = 0.0f;
        referenceFrequency = frequencyEstimates[maxMagnitudeIndex];
        for (i = 0; i < localMaximaIndexes.size(); ++i) {
            if (!(frequencyEstimates[localMaximaIndexes.get(i)] < 0.0f)) continue;
            localMaximaIndexes.remove(i);
            frequencyEstimates[localMaximaIndexes.get((int)i).intValue()] = 1.0f;
            --i;
        }
        for (i = 1; i < localMaximaIndexes.size(); ++i) {
            double centPrev;
            double centCurrent = PitchConverter.hertzToAbsoluteCent(frequencyEstimates[localMaximaIndexes.get(i)]);
            double centDelta = centCurrent - (centPrev = PitchConverter.hertzToAbsoluteCent(frequencyEstimates[localMaximaIndexes.get(i - 1)]));
            if (!(centDelta < (double)minDistanceInCents)) continue;
            if (magnitudes[localMaximaIndexes.get(i)] > magnitudes[localMaximaIndexes.get(i - 1)]) {
                localMaximaIndexes.remove(i - 1);
            } else {
                localMaximaIndexes.remove(i);
            }
            --i;
        }
        float[] maxMagnitudes = new float[localMaximaIndexes.size()];
        for (int i2 = 0; i2 < localMaximaIndexes.size(); ++i2) {
            maxMagnitudes[i2] = magnitudes[localMaximaIndexes.get(i2)];
        }
        Arrays.sort(maxMagnitudes);
        float peakthresh = maxMagnitudes[0];
        if (maxMagnitudes.length > numberOfPeaks) {
            peakthresh = maxMagnitudes[maxMagnitudes.length - numberOfPeaks];
        }
        for (Integer i3 : localMaximaIndexes) {
            if (!(magnitudes[i3] >= peakthresh)) continue;
            float frequencyInHertz = frequencyEstimates[i3];
            float binMagnitude = magnitudes[i3];
            SpectralPeak peak = new SpectralPeak(0.0f, frequencyInHertz, binMagnitude, referenceFrequency, i3);
            spectralPeakList.add(peak);
        }
        return spectralPeakList;
    }

    public static final float median(double[] arr) {
        return SpectralPeakProcessor.percentile(arr, 0.5);
    }

    public static final float percentile(double[] arr, double p) {
        if (p < 0.0 || p > 1.0) {
            throw new IllegalArgumentException("Percentile out of range.");
        }
        Arrays.sort(arr);
        double t = p * (double)(arr.length - 1);
        int i = (int)t;
        return (float)(((double)(i + 1) - t) * arr[i] + (t - (double)i) * arr[i + 1]);
    }

    public static double median(float[] m) {
        Arrays.sort(m);
        int middle = m.length / 2;
        if (m.length % 2 == 1) {
            return m[middle];
        }
        return (double)(m[middle - 1] + m[middle]) / 2.0;
    }

    public static class SpectralPeak {
        private final float frequencyInHertz;
        private final float magnitude;
        private final float referenceFrequency;
        private final int bin;
        private final float timeStamp;

        public SpectralPeak(float timeStamp, float frequencyInHertz, float magnitude, float referenceFrequency, int bin) {
            this.frequencyInHertz = frequencyInHertz;
            this.magnitude = magnitude;
            this.referenceFrequency = referenceFrequency;
            this.timeStamp = timeStamp;
            this.bin = bin;
        }

        public float getRelativeFrequencyInCents() {
            if (this.referenceFrequency > 0.0f && this.frequencyInHertz > 0.0f) {
                float refInCents = (float)PitchConverter.hertzToAbsoluteCent(this.referenceFrequency);
                float valueInCents = (float)PitchConverter.hertzToAbsoluteCent(this.frequencyInHertz);
                return valueInCents - refInCents;
            }
            return 0.0f;
        }

        public float getTimeStamp() {
            return this.timeStamp;
        }

        public float getMagnitude() {
            return this.magnitude;
        }

        public float getFrequencyInHertz() {
            return this.frequencyInHertz;
        }

        public float getRefFrequencyInHertz() {
            return this.referenceFrequency;
        }

        public String toString() {
            return String.format("%.2f %.2f %.2f", Float.valueOf(this.frequencyInHertz), Float.valueOf(this.getRelativeFrequencyInCents()), Float.valueOf(this.magnitude));
        }

        public int getBin() {
            return this.bin;
        }
    }
}

